--[[
    GloabalI18N.lua

    Adds the Mod l10n entries to the global I18N table

	@author: 		BayernGamers
	@date: 			27.03.2025
	@version:		1.0

	History:		v1.0 @27.03.2025 - initial implementation in FS 25
                    ------------------------------------------------------------------------------------------------------
	
	License:        Terms:
                        Usage:
                            Feel free to use this work as-is as long as you adhere to the following terms:
						Attribution:
							You must give appropriate credit to the original author when using this work.
						No Derivatives:
							You may not alter, transform, or build upon this work in any way.
						Usage: 
							The work may be used for personal and commercial purposes, provided it is not modified or adapted.
						Additional Clause:
							This script may not be converted, adapted, or incorporated into any other game versions or platforms except by GIANTS Software.
]]
source(Utils.getFilename("scripts/utils/LoggingUtil.lua", g_currentModDirectory))

local log = LoggingUtil.new(true, LoggingUtil.DEBUG_LEVELS.HIGH, "GlobalI18N.lua")

GlobalI18N = {}
GlobalI18N.MOD_DIR = g_currentModDirectory
GlobalI18N.MOD_NAME = g_currentModName
GlobalI18N.modDescPath = GlobalI18N.MOD_DIR .. "modDesc.xml"

function GlobalI18N.load() 
    log:printDevInfo("Loading mod l10n entries globally", LoggingUtil.DEBUG_LEVELS.HIGH)
    local xmlFile = XMLFile.load("ModFile", GlobalI18N.modDescPath)

    xmlFile:iterate("modDesc.l10n.text", function (_, baseName)
        local name = xmlFile:getString(baseName .. "#name")
        local text = xmlFile:getString(baseName .. "." .. g_languageShort)

        if text == nil then
            text = xmlFile:getString(baseName .. ".en")

            if text == nil then
                text = xmlFile:getString(baseName .. ".de")
            end
        end

        if text == nil or name == nil then
            print("Warning: No l10n text found for entry '" .. name .. "' in mod '" .. GlobalI18N.MOD_NAME .. "'")
        elseif g_i18n:hasText(GlobalI18N.MOD_NAME .. name) then
            print("Warning: Duplicate l10n entry '" .. name .. "' in mod '" .. GlobalI18N.MOD_NAME .. "'. Ignoring this definition.")
        else
            g_i18n:setGlobalText(name, text)
        end
    end)

    local l10nFilenamePrefix = xmlFile:getString("modDesc.l10n#filenamePrefix")

    if l10nFilenamePrefix ~= nil then
        local l10nFilenamePrefixFull = Utils.getFilename(l10nFilenamePrefix, GlobalI18N.MOD_DIR)
        local l10nXmlFile, l10nFilename = nil
        local langs = {
            g_languageShort,
            "en",
            "de"
        }

        for _, lang in ipairs(langs) do
            l10nFilename = l10nFilenamePrefixFull .. "_" .. lang .. ".xml"

            if fileExists(l10nFilename) then
                l10nXmlFile = loadXMLFile("modL10n", l10nFilename)

                break
            end
        end

        if l10nXmlFile ~= nil then
            local textI = 0

            while true do
                local key = string.format("l10n.texts.text(%d)", textI)

                if not hasXMLProperty(l10nXmlFile, key) then
                    break
                end

                local name = getXMLString(l10nXmlFile, key .. "#name")
                local text = getXMLString(l10nXmlFile, key .. "#text")

                if name ~= nil and text ~= nil then
                    if g_i18n:hasText(GlobalI18N.MOD_NAME .. name) then
                        print("Warning: Duplicate l10n entry '" .. name .. "' in '" .. l10nFilename .. "'. Ignoring this definition.")
                    else
                        g_i18n:setGlobalText(name, text:gsub("\r\n", "\n"))
                    end
                end

                textI = textI + 1
            end

            textI = 0

            while true do
                local key = string.format("l10n.elements.e(%d)", textI)

                if not hasXMLProperty(l10nXmlFile, key) then
                    break
                end

                local name = getXMLString(l10nXmlFile, key .. "#k")
                local text = getXMLString(l10nXmlFile, key .. "#v")

                if name ~= nil and text ~= nil then
                    if g_i18n:hasText(GlobalI18N.MOD_NAME .. name) then
                        print("Warning: Duplicate l10n entry '" .. name .. "' in '" .. l10nFilename .. "'. Ignoring this definition.")
                    else
                        g_i18n:setGlobalText(name, text:gsub("\r\n", "\n"))
                    end
                end

                textI = textI + 1
            end

            delete(l10nXmlFile)
        else
            print("Warning: No l10n file found for '" .. l10nFilenamePrefix .. "' in mod '" .. GlobalI18N.MOD_NAME .. "'")
        end
    end
end

GlobalI18N.load() 